-------------------------------------------------------------------------------
-- Installation script for Snowcap database
-- PRECONDITION: snowcap database is already created
-- NOTE: this script does NOT grant any additional user rights to the database.
--       that should be done in a separate script.
-------------------------------------------------------------------------------

use [{SNOWCAP_DBNAME}]
go

--------------------------------------------------
-- Main RMD table - contains all detected hosts
-- 	TODO: document that the empty string is "undefined" for NOT NULL entries
--	TODO: add a "comments" column
--------------------------------------------------
create table [dbo].[Hosts] (
	[HostID]			int IDENTITY (1,1) 	NOT NULL,  --TODO: set NOT FOR REPLICATION?
	[MAC]				nchar(12) 			NOT NULL,  --Mac address in hex with no colons
	[OUI]				nchar(6)			NULL,      --First three bytes of MAC, with no colons. should index into OUIs table
	[NetworkAddress]	varchar(50)			NOT NULL,  --NOTE: ipv6 needs 39 characters.
	[NetworkProtocolID]	int					NOT NULL,  --FK into NetworkProtocols table
	[NetbiosName]		nvarchar(16)		NOT NULL,  --netbios name
	[DnsName]			nvarchar(255)		NOT NULL,  --dns name
	[LastDetectTime]	datetime			NULL,
	[NetworkID]			int					NULL,  	   --FK into Networks table
	[Exception]			bit					NOT NULL,  --is this machine an exception? (1=yes, 0=no)
	[EpoIndex]			int					NULL,	   --if non-null, this is a FK into ePO's LeafNode.AutoID
	[Comments]			nvarchar(255)		NOT NULL,   --user comments
	[OSPlatform]		nvarchar(25)		NOT NULL,   --operating system platform
	[OSFamily]			nvarchar(128)		NOT NULL,   --operating system family
	[OSVersion]			nvarchar(128)		NOT NULL,   --operating system version
    [NetBiosComment]    nvarchar(100)       NOT NULL,   --netbios comment
	[Deleted]			bit					NOT NULL,   --has the host been deleted? (1=yes, 0=no)
	[FirstDetectTime]	datetime			NULL,       --first time this host was detected
    [Users]             nvarchar(128)       NOT NULL,   --logged in users
    [Domain]            nvarchar(16)        NOT NULL,   --machine's domain
    [SID]               nvarchar(48)        NOT NULL,   --machine's SID. Max size of 192 bit. Stored as hex.
	[FirstRogueTime]	datetime			NULL,       --First time this machine was detected as rogue
	[RogueType]         int                 NOT NULL,   --Why is this machine considered rogue?
	[LastAgentCom]      datetime            NULL,       --Last time the agent communicated with the epo server.
	[IpAddressNum]      int                 NOT NULL,   --Numeric value of IP address
	[FriendlyName]	    nvarchar(255)		NOT NULL,   --Uses first non-empty value in list: NB Name, DNS Name, IP, MAC
    [MarkedForAction]   bit                 NOT NULL,   --Has this detected machine been marked for future action by the user?
) on [PRIMARY]
-- -- go

--------------------------------------------------
-- All known networks (subnets)
--------------------------------------------------
create table [dbo].[Networks] (
	[NetworkID]			int IDENTITY (1,1) 	NOT NULL,  --TODO: set NOT FOR REPLICATION?
	[NetworkName]       nvarchar(255)	NOT NULL,	-- unique user-friendly name for the sensor
	[SubnetMask]		nvarchar(25)	NOT NULL,   --subnet mask TODO: should this be # of mask bits?
	[NetworkAddress]	nvarchar(25)	NOT NULL,   --IP address prefix, with zeros where the mask doesn't cover
    [Covered]           int             NOT NULL,   -- used to remember whether "subnet uncovered" event was fired for this subnet.
    [NetworkAddressNum]	int     		NOT NULL,   --Numeric value of IP address
) on [PRIMARY]
-- -- go

--------------------------------------------------
-- All known sensors
--------------------------------------------------
create table [dbo].[Sensors] (
	[SensorID]			int IDENTITY (1,1) 	NOT NULL,  --TODO: set NOT FOR REPLICATION?
    [NetworkID]         int                 NOT NULL,  --FK into Networks; network this sensor is monitoring
	[MAC]				nchar(12) 			NOT NULL,  --Mac address in hex with no colons
	[SensorName]	    nvarchar(255)       NOT NULL,  --Host name
	[IPAddress]	        varchar(50)			NOT NULL,  --IP address of sensor
	[OSPlatform]		nvarchar(25)		NOT NULL,  --operating system platform
	[OSFamily]			nvarchar(25)		NOT NULL,  --operating system family
	[OSVersion]			nvarchar(25)		NOT NULL,  --operating system version
	[LastCommunicationTime]	datetime	    NOT NULL,  --last time we heard from this sensor
    [Domain]            nvarchar(16)        NOT NULL,  --machine's domain
    [SID]               nvarchar(48)        NOT NULL,  --machine's SID. Max size of 192 bit. Stored as hex.
	[AgentGUID]		    nchar(36)    		NOT NULL,  --agent guid associated with action
) on [PRIMARY]

--------------------------------------------------
-- MAC OUI table
--------------------------------------------------
create table [dbo].[OUIs] (
	[OrgName]			nvarchar(255)	NOT NULL,	-- Organization name
	[OUI]				nchar(6)		NOT NULL,	-- MAC address prefix
) on [PRIMARY]
-- -- go

--------------------------------------------------
-- Responses
--------------------------------------------------
create table [dbo].[Responses] (
	[ResponseID]		int IDENTITY (1,1) 	NOT NULL,  --TODO: set NOT FOR REPLICATION?
	[Name]		        nvarchar(255)	    NOT NULL,  -- user-specified name of this response
	[Enabled]		    bit				    NOT NULL,  -- 1=response enabled; 0=disabled
	[Condition]	        nvarchar(4000)	    NOT NULL,  -- condition under which this response will fire
	[EventType]	        int                 NOT NULL,  -- Type of event that this response will be evaluated against.
	[ResponseOrder]     int                 NOT NULL,  -- Display and firing order of response.
) on [PRIMARY]
-- -- go

--------------------------------------------------
-- Actions
--------------------------------------------------
create table [dbo].[Actions] (
	[ActionID]		    int IDENTITY (1,1) 	NOT NULL,  --TODO: set NOT FOR REPLICATION?
	[ResponseID]		int				NOT NULL,	-- FK into Responses; indicates which one this is associated with
	[Type]		        nvarchar(50)	NOT NULL,	-- indicates what kind of action this is
) on [PRIMARY]
-- -- go

--------------------------------------------------
-- ActionSettings
--------------------------------------------------
create table [dbo].[ActionSettings] (
	[ActionID]		    int             NOT NULL,   -- FK into Actions; indicates which action these settings are for
	[Name]	            nvarchar(255)	NOT NULL,	-- name of this config parameter
	[Value] 	        nvarchar(4000)	NOT NULL,	-- value for this parameter
) on [PRIMARY]
-- -- go


--------------------------------------------------
-- ActionClasses - used to create actions based on type name
--------------------------------------------------
create table [dbo].[ActionClasses] (
	[Type]		        varchar(50)		NOT NULL,	-- indicates what kind of action this is
	[Class]		        varchar(255)    NOT NULL,	-- indicates the java class that implements this type
) on [PRIMARY]
-- -- go

--------------------------------------------------
-- Contact
--------------------------------------------------
if not exists (select * from [{SNOWCAP_DBNAME}].dbo.sysobjects where id = object_id(N'[dbo].[Contact]'))
create table [dbo].[Contact] (
	[ContactId]    int IDENTITY (1,1) 	    NOT NULL,  --TODO: set NOT FOR REPLICATION?
	[Name]              nvarchar(255)	    NOT NULL,  -- User-friendly display name for this person
	[EmailAddress]	    nvarchar(255)	    NOT NULL,  -- Email address
) on [PRIMARY]
-- -- go


--------------------------------------------------
-- Tools
--------------------------------------------------
if not exists (select * from [{SNOWCAP_DBNAME}].dbo.sysobjects where id = object_id(N'[dbo].[Tools]'))
BEGIN
create table [dbo].[Tools] (
	[ToolID]		    int IDENTITY (1,1) PRIMARY KEY NOT NULL,  --TODO: set NOT FOR REPLICATION?
	[ExeName]           nvarchar(255)	    NOT NULL,  -- Name of the tool (maps to full path in text file)
	[DisplayName]       nvarchar(255)	    NOT NULL,  -- User-defined name for this tool
	[CmdLine]	        nvarchar(255)	    NOT NULL,  -- Command line to be passed to the tool
	[AppAssociation]    int                 NOT NULL,  -- Used to associate this tool with one of the components 1=snowcap, 2=alerting.
) on [PRIMARY]

END
-- -- go

--------------------------------------------------
-- ActionStatus
--------------------------------------------------
CREATE TABLE [dbo].[ActionStatus] (
	[ActionStatusID]    [int] IDENTITY (1, 1)   NOT NULL , --TODO: set NOT FOR REPLICATION?
	[EventLogId]        [int]                   NOT NULL , -- Foreign key to the event that generated the action.
	[TargetHostId]      [int]                   NULL ,     -- Foreign key to the target host of this action
	[TargetNetworkId]   [int]                   NULL ,     -- Foreign key to the target network of this action
	[TargetSensorId]    [int]                   NULL ,     -- Foreign key to the target sensor of this action
	[Type]              [nvarchar] (128)        NOT NULL , -- A string identifier for the type of action
	[Status]            [int]                   NULL ,     -- -1=in progress, 0=success, >0 = error #
	[StartTime]         [datetime]              NULL ,     -- When the action begun
	[EndTime]           [datetime]              NULL ,     -- When the action ended
	[Endkey]            [nvarchar] (128)        NULL ,     -- Action specific key to identify when complete (AgentGUID)
	[Archive]           [bit]                   NULL ,     -- Tells whether this item should be displayed in the UI
	[ActionOutput]      [image]                 NULL,      -- The output of the action (stdout/stderr of a tool)
	[ActionDetail]      [nvarchar] (512)        NULL       -- A detail string relating to the instantiated action (like the commandline for a tool)
) ON [PRIMARY] TEXTIMAGE_ON [PRIMARY]
-- -- go

---------------------------------------------------------
-- EventLog
---------------------------------------------------------
create table [dbo].[EventLog] (
    [EventLogID]        [int] IDENTITY (1, 1)   NOT NULL, --TODO: set NOT FOR REPLICATION?
    [CreateTime]        [datetime]              NOT NULL, -- The time that this event was created
    [Type]              [int]                   NOT NULL, -- The type of event: 1=NewRogue, 2=UserRequest, 3=SubnetUncovered, 4=SensorPushFailed, 5=AgentPushFailed
	[SourceHostId]      [int]                   NULL,     -- Foreign key to the target host of this action
	[SourceNetworkId]   [int]                   NULL,     -- Foreign key to the target network of this action
	[SourceSensorId]    [int]                   NULL,     -- Foreign key to the target sensor of this action
	[Archive]           [bit]                   NULL,     -- Tells whether this item should be displayed in the UI
	[ActionCount]       [int]                   NOT NULL,
) on [PRIMARY]

--------------------------------------------------
-- Configuration
--------------------------------------------------
create table [dbo].[Configuration] (
	[SchemaVersion]		    int			  NOT NULL,	  -- helps when we need to upgrade schema
	[EmailServer]		    nvarchar(255) NOT NULL,	  -- server to send email to.
	[EmailFromAddr] 	    nvarchar(255) NOT NULL,	  -- address that should appear in "from" field
	[EmailFromName]	        nvarchar(255) NOT NULL,	  -- display name that should appear in "from" field
	[PageCount]		        int			  NOT NULL,
	[SortColumn1]           int           NOT NULL,   -- column machines are first sorted on, 0=none
	[SensorWindow]          decimal(12)   NOT NULL,   -- number of minutes before sensor is considered "down"
	[EpoImportPeriod]       decimal(12)   NOT NULL,   -- how often to import data from ePO, in hours
	[MachineWindow]         decimal(12)   NOT NULL,   -- number of minutes before machine is considered "inactive"
	[SensorInstallDir]      nvarchar(255) NOT NULL,	  -- directory sensor will be installed in
	[AgentWakeup]           int           NOT NULL,   -- 1=perform agent wakeup, 0=don't
	[PushRandomization]     int           NOT NULL,   -- randomization time in minutes for Sensor push
	[EpoDbLookupMethod]     int           NOT NULL,   -- how should detected hosts be matched to ePO database?
	[AutoRefreshPeriod]     decimal(12)   NOT NULL,   -- number of seconds until next auto refresh of UI list pages
	[AutoRefreshEnabled]    int           NOT NULL,   -- is auto refresh enabled, 0=no, 1=yes
	[DeadAgentPeriod]       decimal(12)   NOT NULL,   -- in minutes; machine is rogue if this is >0
	                                                  --   and agent hasn't phoned home in this period
    [SensorDeployAuto]      nvarchar(255) NOT NULL,   -- [auto | manual]
    [NumSensorsPerSubnet]   int           NOT NULL,   -- number of sensors desired per subnet for auto selection
    [SensorDeployPref]      nvarchar(255) NOT NULL,   -- sensor auto deployment selection criteria preferences
    [SensorDeployHostOp]    nvarchar(255) NOT NULL,   -- sensor auto deployment hostname operation
    [SensorDeployHostValue] nvarchar(255) NOT NULL,   -- sesnor auto deployment hostname value

    [LimitSensorsEnabled]   bit           NOT NULL,   -- is limiting active sensors enabled?
    [MaxActiveSensors]      int           NOT NULL,   -- max active sensors per subnet
    [MaxActiveSensorTime]   decimal(12)   NOT NULL,   -- in hours, how long an active sensor can be active before it's rotated out
    [GracePeriodEnabled]    bit           NOT NULL,   -- is the rogue machine grace period enabled?
    [GracePeriod]           decimal(12)   NOT NULL,   -- rogue machine grace period, in minutes
    [AdditionalAgentPorts]  nvarchar(255) NOT NULL,   -- ports to check for ePO agent in addition to the one listed in the server.ini
    [MachineListCols]       nvarchar(255) NOT NULL,   -- columns displayed in machine list
    [SubnetListCols]        nvarchar(255) NOT NULL,   -- columns displayed in subnet list
    [EventsListCols]        nvarchar(255) NOT NULL,   -- columns displayed in the events list
    [ActionStatusListCols]  nvarchar(255) NOT NULL,   -- columns displayed in the action status list
    [MLColsDefault]         nvarchar(255) NOT NULL,   -- default columns displayed in machine list if problems arise
    [SLColsDefault]         nvarchar(255) NOT NULL,   -- default columns displayed in subnet list if problems arise
    [ELColsDefault]         nvarchar(255) NOT NULL,   -- default columns displayed in the events list if problems arise
    [ASColsDefault]         nvarchar(255) NOT NULL,   -- default columns displayed in the action status list if problems arise.
    [AllowSiteAdmins]       int           NOT NULL,   -- whether or not site administrators are allowed to modify external tools, etc.

) on [PRIMARY]
go

--------------------------------------------------
-- Indexes
--------------------------------------------------

create unique clustered index [IX_Hosts_MAC] on [dbo].[Hosts]([MAC]) on [PRIMARY]
go

create unique clustered index [IX_Networks] on [dbo].[Networks]([NetworkAddress],[SubnetMask]) on [PRIMARY]
go

create unique clustered index [IX_Sensors] on [dbo].[Sensors]([MAC]) on [PRIMARY]
go

create clustered index [IX_ActionStatus] on [dbo].[ActionStatus]([EndKey]) on [PRIMARY]
go

create clustered index [IX_ActionSettings] on [dbo].[ActionSettings]([ActionID]) on [PRIMARY]
go

--------------------------------------------------
-- Hosts Table Indexes
--------------------------------------------------
create index [IX_Hosts_OUI] on [dbo].[Hosts]([OUI]) on [PRIMARY]
go

create index [IX_Hosts_NetbiosName] on [dbo].[Hosts]([NetbiosName]) on [PRIMARY]
go

create index [IX_Hosts_DnsName] on [dbo].[Hosts]([DnsName]) on [PRIMARY]
go

create index [IX_Hosts_LastDetectTime] on [dbo].[Hosts]([LastDetectTime]) on [PRIMARY]
go

create index [IX_Hosts_Comments] on [dbo].[Hosts]([Comments]) on [PRIMARY]
go

create index [IX_Hosts_OSPlatform] on [dbo].[Hosts]([OSPlatform]) on [PRIMARY]
go

create index [IX_Hosts_OSFamily] on [dbo].[Hosts]([OSFamily]) on [PRIMARY]
go

create index [IX_Hosts_OSVersion] on [dbo].[Hosts]([OSVersion]) on [PRIMARY]
go

create index [IX_Hosts_NetBiosComment] on [dbo].[Hosts]([NetBiosComment]) on [PRIMARY]
go

create index [IX_Hosts_FirstDetectTime] on [dbo].[Hosts]([FirstDetectTime]) on [PRIMARY]
go

create index [IX_Hosts_Users] on [dbo].[Hosts]([Users]) on [PRIMARY]
go

create index [IX_Hosts_Domain] on [dbo].[Hosts]([Domain]) on [PRIMARY]
go

create index [IX_Hosts_FirstRogueTime] on [dbo].[Hosts]([FirstRogueTime]) on [PRIMARY]
go

create index [IX_Hosts_RogueType] on [dbo].[Hosts]([RogueType]) on [PRIMARY]
go

create index [IX_Hosts_LastAgentCom] on [dbo].[Hosts]([LastAgentCom]) on [PRIMARY]
go

create index [IX_Hosts_IpAddressNum] on [dbo].[Hosts]([IpAddressNum]) on [PRIMARY]
go

create index [IX_Hosts_FriendlyName] on [dbo].[Hosts]([FriendlyName]) on [PRIMARY]
go

--------------------------------------------------
-- Networks Table Indexes
--------------------------------------------------

create index [IX_Networks_NetworkAddressNum] on [dbo].[Networks]([NetworkAddressNum]) on [PRIMARY]
go

create index [IX_Networks_NetworkName] on [dbo].[Networks]([NetworkName]) on [PRIMARY]
go

--------------------------------------------------
-- ActionStatus Table Indexes
--------------------------------------------------

create index [IX_ActionStatus_Type] on [dbo].[ActionStatus]([Type]) on [PRIMARY]
go

create index [IX_ActionStatus_Status] on [dbo].[ActionStatus]([Status]) on [PRIMARY]
go

create index [IX_ActionStatus_StartTime] on [dbo].[ActionStatus]([StartTime]) on [PRIMARY]
go

create index [IX_ActionStatus_EndTime] on [dbo].[ActionStatus]([EndTime]) on [PRIMARY]
go


--------------------------------------------------
-- EventLog Table Indexes
--------------------------------------------------
create index [IX_EventLog_CreateTime] on [dbo].[EventLog]([CreateTime]) on [PRIMARY]
go

--------------------------------------------------
-- Constraints
-- TODO: which PKs want to be clustured?
--		 I claim none of them ... is this right?
--------------------------------------------------

alter table [dbo].[Networks] add
	constraint [PK_Networks] primary key nonclustered ([NetworkID]) on [PRIMARY],
	constraint [DF_Networks_Covered] default 1 for [Covered]
go

alter table [dbo].[OUIs] add
	constraint [PK_OUI] primary key ([OUI]) on [PRIMARY]
go

alter table [dbo].[Hosts] add
	constraint [PK_Hosts]     		primary key nonclustered ([HostID]) on [PRIMARY],
	constraint [FK_Hosts_Networks]  foreign key ([NetworkID])    references [Networks]([NetworkID])
go

alter table [dbo].[Sensors] add
	constraint [PK_Sensors]    		    primary key nonclustered ([SensorID]) on [PRIMARY],
	constraint [FK_Sensors_Networks] 	foreign key ([NetworkID]) references [Networks]([NetworkID])
go

alter table [dbo].[Responses] add
	constraint [PK_Responses] primary key ([ResponseID]) on [PRIMARY],
	constraint [UN_Responses] unique ([ResponseOrder])
go

alter table [dbo].[Actions] add
	constraint [PK_Actions] primary key ([ActionID]) on [PRIMARY],
	constraint [FK_Actions_Responses] foreign key ([ResponseID]) references [Responses]([ResponseID])
go

alter table [dbo].[ActionSettings] add
	constraint [FK_ActionSettings] foreign key ([ActionID]) references [Actions]([ActionID])
go

alter table [dbo].[ActionStatus] add
    constraint [PK_ActionStatus] primary key nonclustered ([ActionStatusID]) on [PRIMARY],
	CONSTRAINT [DF_ActionStatus_Status] DEFAULT ((-1)) FOR [Status],
	CONSTRAINT [FK_ActionStatus_Hosts] FOREIGN KEY ([TargetHostId]) REFERENCES [dbo].[Hosts] ([HostID]),
	CONSTRAINT [FK_ActionStatus_Networks] FOREIGN KEY ([TargetNetworkId]) REFERENCES [dbo].[Networks] ([NetworkID]),
	CONSTRAINT [FK_ActionStatus_Sensors] FOREIGN KEY ([TargetSensorId]) REFERENCES [dbo].[Sensors] ([SensorID])
go

alter table [dbo].[EventLog] add
    constraint [PK_EventLog] primary key nonclustered ([EventLogID]) on [PRIMARY],
	CONSTRAINT [FK_EventLog_Hosts] FOREIGN KEY ([SourceHostId]) REFERENCES [dbo].[Hosts] ([HostID]),
	CONSTRAINT [FK_EventLog_Networks] FOREIGN KEY ([SourceNetworkId]) REFERENCES [dbo].[Networks] ([NetworkID]),
	CONSTRAINT [FK_EventLog_Sensors] FOREIGN KEY ([SourceSensorId]) REFERENCES [dbo].[Sensors] ([SensorID])
go

ALTER TABLE [dbo].[Contact] ADD
	constraint [PK_Contact] primary key ([ContactId]) on [PRIMARY]
go

--------------------------------------------------
-- Initial values
--------------------------------------------------

-- configuration info
insert into [dbo].[Configuration] values (
    63,       -- schema version
    'mail-server',
    'from@example.com',
    'Snowcap Server',
    15,
    0,
    5400000,    -- SensorWindow
    10800000,   -- EpoImportPeriod
    259200000,  -- MachineWindow
    'C:\McAfee',
    1,
    1,
    1,
    30000,      -- AutoRefreshPeriod
    1,
    604800000,  -- DeadAgentPeriod (7 days)
    'auto',
    3,
    '03',
    0,
    'server',
    0,          -- LimitSensorsEnabled
    2,          -- MaxActiveSensors
    43200000,   -- MaxActiveSensorTime (hours)
    0,          -- GracePeriodEnabled
    3600000,    -- GracePeriod (minutes)
    '',         -- No default additional agent port
    '29:23:27:26:4:8', -- columns for the machine list
    '9:10:2:7:8',      -- columns for the subnet list
    '14:2:9:15:8',    -- columns for the events list
    '26:20:24:8:9',    -- columns for the action status list
    '29:23:27:26:4:8', -- default columns for the machine list
    '9:10:2:7:8',      -- default columns for the subnets list
    '14:2:8:15:12',    -- default columns for the events list
    '26:20:24:8:9',     -- default columns for the action status list.
    1                   -- allow site admins
    )

-- registered action classes...
insert into [dbo].[ActionClasses] values ('smtp',            'com.nai.mcafee.snowcap.action.concrete.EmailAction')
insert into [dbo].[ActionClasses] values ('except',          'com.nai.mcafee.snowcap.action.concrete.ExceptHostAction')
insert into [dbo].[ActionClasses] values ('unexcept',        'com.nai.mcafee.snowcap.action.concrete.UnexceptHostAction')
insert into [dbo].[ActionClasses] values ('removehost',      'com.nai.mcafee.snowcap.action.concrete.RemoveHostAction')
insert into [dbo].[ActionClasses] values ('external tool',   'com.nai.mcafee.snowcap.action.concrete.ExternalToolAction')
insert into [dbo].[ActionClasses] values ('pushAgent',       'com.nai.mcafee.snowcap.action.concrete.PushEPOAgentAction')
insert into [dbo].[ActionClasses] values ('add_to_epo',      'com.nai.mcafee.snowcap.action.concrete.AddToEPOTreeAction')
insert into [dbo].[ActionClasses] values ('markforaction',   'com.nai.mcafee.snowcap.action.concrete.MarkForAction')
insert into [dbo].[ActionClasses] values ('unmarkforaction', 'com.nai.mcafee.snowcap.action.concrete.UnmarkForAction')
insert into [dbo].[ActionClasses] values ('epoagent_query',  'com.nai.mcafee.snowcap.action.concrete.QueryEPOAgentAction')
insert into [dbo].[ActionClasses] values ('send_event',      'com.nai.mcafee.snowcap.action.concrete.SendEpoServerEventAction')

go
