#!/usr/bin/python
# Copyright (C) 2004 Networks Associates Technology Inc. All rights reserved.
#
# AntiSpam Evaluation manager
#
# antispam-eval.py [start | check]
#
# Args:
#
# start
#   if evaluation has been run before, exit with -1
#   otherwise start evaluation, run all necessary scripts, print number of days
#
#
# check
#   prints number of days remaining,
#
#
# sanitycheck
#   Checks the state of the installation according to the magic files, and ensures
#   WebShield.xml is kept in synch. This should be called after restoring from
#   configuration.
#
# (no args):
#   if evaluation has expired, remove it and send log
#   if run on same day of week as it was installed, send warning
#   if more than 30 days remain, re-install evaluation
#
import time, sys, os, exceptions, string, libxml2

#                           ui_enable   ui_eval    ui_canEval   evalFile
CanEvaluate  = 0        #       N           N           X           N
FullInstall  = 1        #       X           N           N           N
Evaluating   = 2        #       X           X           N           X
NotInstalled = 3        #       N           N           N           X

# evaluation period
EvalDays     = 30


try:
	WSBIN		= os.environ["WSBIN"]
	evrep		= os.path.join(WSBIN, "evrep") + " -n spamkiller_eval_monitor -e SPAM_INSTALL_STATE::"

	XmlConfDir	= os.environ['XMLCONFDIR']
	evalFile	= os.path.join(XmlConfDir, "spamconfig/antispam-eval.log")
	ui_enable	= os.path.join(XmlConfDir, "spamconfig/wsui_spam_activation")
	ui_eval		= os.path.join(XmlConfDir, "spamconfig/wsui_spam_evaluation")
	ui_canEval	= os.path.join(XmlConfDir, "spamconfig/wsui_spam_can_evaluate")
	MACHINE_XML	= os.path.join(XmlConfDir, "machine.xml")
	SYSTEM_XML	= os.path.join(XmlConfDir, "system.xml")
	SMTP_XML	= os.path.join(XmlConfDir, "smtp.xml")
	CRONLINK	= "/etc/cron.daily/spameval"

except KeyError:
	# the environment variables are not correctly set so rerun the scripts with the correct environment
	cmd = string.join(sys.argv, " ")
	sys.exit(os.system(". /var/NAIENV/.profile.vars; "+cmd))

class ExpiredEx(exceptions.Exception):
	pass

class NotInstalledEx(exceptions.Exception):
	pass

class FullInstallEx(exceptions.Exception):
	pass

class AlreadyRunEx(exceptions.Exception):
	pass

class Eval:
	def __init__(self):
		self.state = CanEvaluate
		self.epochNow      = time.time()
		self.warning       = False              # Is today a day to send out warnings?
		self.epochEnd      = self.epochNow
		self.daysRemaining = 0

		evalFileExists      = (os.access(evalFile,   os.F_OK) != 0)
		uiEnableFileExists  = (os.access(ui_enable,  os.F_OK) != 0)
		uiEvaluationExists  = (os.access(ui_eval,    os.F_OK) != 0)
		uiCanEvalFileExists = (os.access(ui_canEval, os.F_OK) != 0)

		stateOk = True
		if uiEnableFileExists:
			if uiEvaluationExists:
				if evalFileExists and not uiCanEvalFileExists:
					self.state = Evaluating
				else:
					stateOk = False
			else:
				if not uiCanEvalFileExists and not evalFileExists:
					self.state = FullInstall
				else:
					stateOk = False
		else:
			if uiCanEvalFileExists:
				if not uiEvaluationExists and not evalFileExists:
					self.state = CanEvaluate
				else:
					stateOk = False
			else:
				if evalFileExists and not uiEvaluationExists:
					self.state = NotInstalled
				else:
					stateOk = False
		if not stateOk:
			import syslog
			syslog.syslog(syslog.LOG_ERR, "spamkiller state inconsistent - treat as not installed")
			self.state = NotInstalled

		if evalFileExists:
			try:
				file       = open(evalFile, "r")
				endStr     = file.readline()
				endTime    = time.strptime(endStr, "%Y-%m-%d")
				endEpoch   = time.mktime(endTime)
				startEpoch = endEpoch - (EvalDays + 0.5) * 24 * 60 * 60
				if time.localtime(startEpoch)[6] == time.localtime(self.epochNow)[6]: # same day of week as installation?
					self.warning = True

				secondsRemaining = endEpoch - self.epochNow
				self.daysRemaining = round(secondsRemaining / 60 / 60 / 24)

			except ValueError:
				self.daysRemaining = 0

			except IOError:
				pass
			except Exception, e:
				print "Internal error"
				print_exception(e)
				self.daysRemaining = 0

	def getDaysRemaining(self):
		if self.state == FullInstall:
			raise FullInstallEx, "No days remaining, full install"
		if self.daysRemaining <= 0:
			return 0
		if self.state == NotInstalled or self.state == CanEvaluate:
			raise NotInstalledEx, "No days remaining, not installed"
		return self.daysRemaining


	def activate(self):
		if self.state == FullInstall:
			raise FullInstallEx, "Cannot activate, full install"
		if self.state == NotInstalled:
			raise AlreadyRunEx, "Cannot activate, already run"
		self.epochEnd = self.epochNow + (EvalDays + 0.5) * 24 * 60 * 60
		tplEnd = time.localtime(self.epochEnd)
		dateEnd = time.strftime("%Y-%m-%d", tplEnd)
		DoW = "%d" % (time.localtime(self.epochNow)[6]) # weekday

		# Do activation ...
		self.evaluateScript()
		os.system(evrep+"EvalStart")
		os.system("/usr/bin/logger 'SpamAssassin starting evaluation'")

		file = open(evalFile, "w")
		file.write(dateEnd)

	def cron(self):
		if self.state == FullInstall:
			raise FullInstallEx, ", full install"
		if self.state == NotInstalled:
			raise AlreadyRunEx, ", already run"

		if self.daysRemaining <= 0:
			# Do deactivation
			os.system(evrep+"EvalExpire")
			os.system("/usr/bin/logger 'SpamAssassin evaluation expired'")
			sk_eval.deactivateScript()
		elif self.daysRemaining > EvalDays + 1:
			# Reset length of evaluation
			self.activate()
		elif self.daysRemaining == 1 or self.warning:
			# Send warning event.
			os.system(evrep+"EvalWaning -i DAYS_LEFT-v %d" % (self.daysRemaining))
			os.system("/usr/bin/logger 'SpamAssassin evaluation will expire in %d days'" % (self.daysRemaining))
			pass

	def SetXML(self, newSpamEvaluationState, newSpamEngineEnabled):
		machineXML = libxml2.parseFile(MACHINE_XML)
		ctxt = machineXML.xpathNewContext()
		SpamEvaluationState = ctxt.xpathEval("//Settings[@name='initialization']/Attr[@name='SpamEvaluationState']/@value")
		if len(SpamEvaluationState) != 1:
			raise RuntimeError, "Unable to find SpamEvaluationState in the XML setting file"
		if SpamEvaluationState[0].content != newSpamEvaluationState:
			SpamEvaluationState[0].setContent(newSpamEvaluationState)
			machineXML.saveFile(MACHINE_XML)

		smtpXML = libxml2.parseFile(SMTP_XML)
		ctxt = smtpXML.xpathNewContext()
		SpamEngineEnabled = ctxt.xpathEval("//Settings[@name='scanning']/Attr[@name='SpamEngineEnabled']/@value")
		if len(SpamEngineEnabled) != 1:
			raise RuntimeError, "Unable to find SpamEngineEnabled in the XML setting file"
		if SpamEngineEnabled[0].content != newSpamEngineEnabled:
			SpamEngineEnabled[0].setContent(newSpamEngineEnabled)
			smtpXML.saveFile(SMTP_XML)

		systemXML = libxml2.parseFile(SYSTEM_XML)
		ctxt = systemXML.xpathNewContext()
		SpamdaemonEnabled = ctxt.xpathEval("//GlobalSettings[@name='system']/Policy[@name='system_variables']/PolicyStatement/Settings[@name='spamdaemon']/Attr[@name='Enabled']/@value")
		if len(SpamdaemonEnabled) != 1:
			raise RuntimeError, "Unable to find spamdaemon Enabled in the XML setting file"
		if SpamdaemonEnabled[0].content != newSpamEngineEnabled:
			SpamdaemonEnabled[0].setContent(newSpamEngineEnabled)
			systemXML.saveFile(SYSTEM_XML)
			if sys.argv[1] != "sanitycheck":
				os.system("/usr/sbin/webshield update-xmlconf >/dev/null 2>&1 &")

	def evaluateScript(self):
		self.SetXML("evaluating", "1")

		for f in [ui_canEval, ui_eval, ui_enable]:
			try:
				os.unlink(f)
			except:
				pass
		for f in [ui_eval, ui_enable]:
			f = open(f, "w")
			f.close()
		os.system("/bin/ln -sf '%s' '%s'" % (os.path.abspath(sys.argv[0]), CRONLINK))

	def resetScript(self):
		self.SetXML("evaluate", "0")

		for f in [ui_canEval, ui_eval, ui_enable, evalFile, CRONLINK]:
			try:
				os.unlink(f)
			except:
				pass
		f = open(ui_canEval, "w")
		f.close()

	def activateScript(self):
		self.SetXML("done", "1")

		for f in [evalFile, ui_canEval, ui_eval, CRONLINK, ui_enable]:
			try:
				os.unlink(f)
			except:
				pass
		f = open(ui_enable, "w")
		f.close()

	def deactivateScript(self):
		self.SetXML("done", "0")

		for f in [ui_canEval, ui_eval, CRONLINK, ui_enable]:
			try:
				os.unlink(f)
			except:
				pass

if __name__ == '__main__':
	sk_eval = Eval()

	if len(sys.argv) == 2:
		if sys.argv[1] == "start":
			try:
				sk_eval.activate()
			except FullInstallEx:
				pass
			except AlreadyRunEx:
				pass
			except NotInstalledEx:
				pass
			except Exception, details:
				print "Unexpected error:", details
				sys.exit(-1)
			sys.exit(0)

		elif sys.argv[1] == "reset":
			try:
				sk_eval.resetScript()
				os.system(evrep+"UnInstalled")
				os.system("/usr/bin/logger 'SpamAssassin uninstalled'")
			except Exception, details:
				print "Unexpected error:", details
				sys.exit(-1)
			sys.exit(0)

		elif sys.argv[1] == "install":
			try:
				sk_eval.activateScript()
				os.system(evrep+"Installed")
				os.system("/usr/bin/logger 'SpamAssassin installed'")
			except FullInstallEx:
				pass
			except AlreadyRunEx:
				pass
			except NotInstalledEx:
				pass
			except Exception, details:
				print "Unexpected error:", details
				sys.exit(-1)
			sys.exit(0)

		elif sys.argv[1] == "stop":
			try:
				sk_eval.deactivateScript()
				os.system(evrep+"UnInstalled")
				os.system("/usr/bin/logger 'SpamAssassin uninstalled'")
			except FullInstallEx:
				pass
			except AlreadyRunEx:
				pass
			except NotInstalledEx:
				pass
			except Exception, details:
				print "Unexpected error:", details
				sys.exit(-1)
			sys.exit(0)

		elif sys.argv[1] == "check":
			try:
				numDays = int(sk_eval.getDaysRemaining())
				print "%d" % numDays
			except FullInstallEx:
				print "Full install"
			except AlreadyRunEx:
				print "0"
			except NotInstalledEx:
				print "0"
			except Exception, details:
				print "Unexpected error:", details
				sys.exit(-1)
			sys.exit(0)

		elif sys.argv[1] == "sanitycheck":
			if sk_eval.state == FullInstall:
				sk_eval.activateScript()
			elif sk_eval.state == NotInstalled:
				sk_eval.deactivateScript()
			elif sk_eval.state == CanEvaluate:
				sk_eval.resetScript()
			elif sk_eval.state == Evaluating:
				sk_eval.evaluateScript()
			else:
				pass

		elif sys.argv[1] == "status":
			state = "CE"
			if sk_eval.state == CanEvaluate:
				state = "Can Evaluate"
			elif sk_eval.state == FullInstall:
				state = "Full Install"
			elif sk_eval.state == Evaluating:
				state = "Evaluating"
			elif sk_eval.state == NotInstalled:
				state = "Not Installed"
			numDays = 0
			try:
				numDays = int(sk_eval.getDaysRemaining())
			except FullInstallEx:
				pass
			except Exception, details:
				print "Unexpected error:", details
				sys.exit(-1)
			print state+",", numDays, "day(s) remaining"
			sys.exit(0)

	else:
		try:
			sk_eval.cron()
		except FullInstallEx:
			pass
		except AlreadyRunEx:
			pass
		except NotInstalledEx:
			pass
		except Exception, details:
			print "Unexpected error:", details
			sys.exit(-1)
