#!/opt/NETAwss/ui/wsadmin/scripts/surun /usr/bin/python
# Copyright (C) 2004 Networks Associates Technology Inc.
# All rights reserved.
import os, sys, libxml2, string, re, ftplib, smtplib, telnetlib, libwsa, timeoutsocket
from libwsa import _TR_
from libwsa import Trace
from libwsa import NewGetStatusOutput

libwsa.do_trace = False
formatXML = False

# TODO: set global timeout on socket operations (smtp, ls)

class CTest(object):
	"""
	Base class for the configuration test
	"""
	ATOMIC = 0
	"""Enumeration value for "basic" tests"""
	GROUP = 1
	"""Enumeration value for groups of tests"""
	
	MACHINE = 0
	"""Enumeration value for machine.xml"""
	NETWORK = 1
	"""Enumeration value for network.xml"""
	SYSTEM = 2
	"""Enumeration value for system.xml"""
	SMTP = 3
	"""Enumeration value for smtp.xml"""
	N_XML_FILES = 4
	"""Number of XML files"""

	XMLDOCS = [ None for x in range(0, N_XML_FILES) ]
	"""Cache for the XML documents"""
	
	SETTING_DIRS = '/config/wsxmlconf'
	"""Path to the config directory"""
	XMLNAMES = [ "machine", "network", "system", "smtp" ]
	"""Names for the XML files"""
	
	def __init__(self, parent, specificID = ''):
		"""
		Constructor
		Params:
			parent: the "parent" test
			specificID: the string to add to the class name in order to get a unique ID
		"""
		self.parent = parent
		self.specificID = specificID
	def getID(self):
		"""Returns the instance ID"""
		ID = ''
		if self.parent != None:
			ID = self.parent.getID() + ':'
		return ID + self.__class__.__name__ + self.specificID
	def getType(self):
		"""Returns the test type"""
		return CTest.ATOMIC
	def getDesc(self):
		"""Returns the test description"""
		return "???"
	def getDoc(self, docID):
		"""
		Returns the XML document 'docID':
		Params:
			docID: the document ID
		"""
		if CTest.XMLDOCS[docID] == None:
			CTest.XMLDOCS[docID] = libxml2.parseFile(os.path.join(CTest.SETTING_DIRS, CTest.XMLNAMES[docID])+".xml")
		return CTest.XMLDOCS[docID]
	def AddDescTo(self, parent):
		"""
		Adds its description the XML element 'parent':
		Params:
			parent: the parent XML element
		Returns:
			the created XML element
		"""
		if self.getType() == CTest.ATOMIC:
			child = parent.newChild(None, "test", None)
		else:
			child = parent.newChild(None, "test_group", None)
		child.setProp("id", self.getID())
		child.setProp("desc", self.getDesc())
		return child
	def RunIf (self, testID, parentXML):
		"""
		Run the test if the testID matches this instance
		Return true if the test was run.
		"""
		if testID == self.getID():
			self.Run(parentXML)
			return True
		return False
	def Run(self, parentXML):
		"""
		Run the test
		
		Add a "test" node to parentXML with the attributes:
			id: the test ID
			status: -1 => error, 0 => warning, 1: success
			msg: an additional string that will be displayed on the UI
			
		Return the newly created node
		"""
		child = parentXML.newChild(None, "test", None)
		child.setProp("id", self.getID())
		child.setProp("status", "0")
		child.setProp("msg", "")
		return child

class TestGroup(CTest):
	"""
	Base class for test groups
	"""
	def __init__(self, parent, specificID = ''):
		"""
		Constructor
		Params:
			parent: the "parent" test
			specificID: the string to add to the class name in order to get a unique ID
		"""
		CTest.__init__(self, parent, specificID)
		self.tests = []
	def getType(self):
		"""Returns the test type"""
		return CTest.GROUP
	def AddDescTo(self, parent):
		"""
		Adds its description the XML element 'parent':
		Params:
			parent: the parent XML element
		Returns:
			the created XML element
		"""
		child = CTest.AddDescTo(self, parent)
		for test in self.tests:
			test.AddDescTo(child)
		return child
	def RunIf (self, testID, parentXML):
		"""
		Run the test if the testID matches this instance
		Return true if the test was run.
		"""
		for test in self.tests:
			if test.RunIf(testID, parentXML):
				# found the right test!
				return True
		return False
		
class NetworkTest(CTest):
	"""
	This class allows to factorize test which have a host and description as parameters
	"""
	def __init__(self, host, desc, parent, specificID = ''):
		"""
		Constructor
		Params:
			host: the host address
			desc: the test description
			parent: the "parent" test
			specificID: the string to add to the class name in order to get a unique ID
		"""
		CTest.__init__(self, parent, specificID)
		self.host = host
		self.desc = desc
	def getDesc(self):
		"""Returns the test description"""
		return self.desc

class PingTest(NetworkTest):
	"""
	This test "pings" a given hosts
	"""
	PING_TIMEOUT = 5
	"""Timeout in second for the oping command"""
	PING_COUNT = 2
	"""Number of ping iterations"""
	def Run(self, parentXML):
		"""
		Run the test
		
		Add a "test" node to parentXML with the attributes:
			id: the test ID
			status: -1 => error, 0 => warning, 1: success
			msg: an additional string that will be displayed on the UI
			
		Return the newly created node
		"""
		child = NetworkTest.Run(self, parentXML)
		cmd = "/bin/ping -W %s -c %s %s" % (PingTest.PING_TIMEOUT, PingTest.PING_COUNT, self.host)
		Trace("Running ping test '%s'" % cmd)
		status, res = NewGetStatusOutput(cmd)
		Trace("Test done: (%d, %s)" % (status, res))
		if status == 0:
			# success
			child.setProp("status", "1")
		else:
			# failure: either the host is not there or something on the road refuses ICMP requests
			child.setProp("status", "0")
		child.setProp("msg", res)
		
class ArpTest(NetworkTest):
	"""
	This test look for a given host in the appliance ARP's table
	"""
	ERROR_MSG = "no entry"
	"""The string to look for an error"""
	def Run(self, parentXML):
		child = NetworkTest.Run(self, parentXML)
		cmd = "/sbin/arp %s" % self.host
		Trace("Running arp test '%s'" % cmd)
		status, res = NewGetStatusOutput(cmd)
		Trace("Test done: (%d, %s)" % (status, res))
		# NB: arp returns always 0 even if the search fail
		# so we have to look for the "no entry" string
		if res.find(ArpTest.ERROR_MSG) < 0:
			# success
			child.setProp("status", "1")
		else:
			# failure
			child.setProp("status", "-1")
		child.setProp("msg", res)
		return child

class DigTest(NetworkTest):
	"""
	This test looks for a A record for a given host on a DNS server
	"""
	__CMD1 = 'dig @%s -t %s %s'
	"""The dig command for a specific DNS server"""
	__CMD2 = 'dig -t %s %s'
	"""The dig command"""
	__ANSWER = 'ANSWER SECTION:'
	"""The string indicating the start of the answer section"""
	__TEST_ADDR = "www.nai.com"
	"""The addresses to look for"""
	T_A = ('a', [re.compile("\\s+IN\\s+A\\s+"), re.compile("\\s+IN\\s+CNAME\\s+")])
	"""type A query"""
	T_NS = ('ns', [re.compile("\\s+IN\\s+NS\\s+")])
	"""type NS query"""
	def __init__(self, host, desc, parent, specificID = '', query_type = T_A, test_address = __TEST_ADDR):
		"""
		Constructor
		Params:
			host: the host address
			desc: the test description
			parent: the "parent" test
			query_type: the DNS query type
			test_address: the test address
			specificID: the string to add to the class name in order to get a unique ID
		"""
		NetworkTest.__init__(self, host, desc, parent, specificID)
		self.test_address = test_address
		self.query_type = query_type
	def Run(self, parentXML):
		"""
		Run the test
		
		Add a "test" node to parentXML with the attributes:
			id: the test ID
			status: -1 => error, 0 => warning, 1: success
			msg: an additional string that will be displayed on the UI
			
		Return the newly created node
		"""
		child = NetworkTest.Run(self, parentXML)
		# run the dig command
		if (self.host != None):
			cmd = DigTest.__CMD1 % (self.host, self.query_type[0], self.test_address)
		else:
			cmd = DigTest.__CMD2 % (self.query_type[0], self.test_address)
		Trace("Running dig test '%s'" % cmd)
		status, output = NewGetStatusOutput(cmd)
		Trace("End dig test (%d)" % status)
		# NB: dig returns a non null value only if the host is not a DNS server or cannot be reached
		if status != 0:
			# failure
			child.setProp("status", "-1")
			child.setProp("msg", "The server could not be reached")
		else:
			# look for the answer section
			res = []
			found = False
			lines = output.split('\n')
			for iline in xrange(0, len(lines)):
				if lines[iline].find(DigTest.__ANSWER) >= 0:
					found = True
					break
			if found:
				for xline in xrange(iline+1, len(lines)):
					if lines[xline] == '':
						break
					for pmatch in self.query_type[1]:
						if pmatch.search(lines[xline]) != None:
							res.append(lines[xline])
							break
			if len(res) == 0:
				# no match
				child.setProp("status", "-1")
				child.setProp("msg", "No matching records")
			else:
				child.setProp("status", "1")
				child.setProp("msg", string.join(res,'\n'))
		return child

class SmtpTest(NetworkTest):
	"""
	This test try a HELO on an SMTP server
	"""
	DEFAULT_DOMAIN = "webshield.nai.com"
	
	def __init__(self, host, desc, parent, port = smtplib.SMTP_PORT ,domain = DEFAULT_DOMAIN, specificID = ''):
		"""
		Constructor
		Params:
			host: the SMTP server address
			desc: the test description
			parent: the "parent" test
			port: the SMTP server port
			domain: the domain name used for the HELO command
			specificID: the string to add to the class name in order to get a unique ID
		"""
		NetworkTest.__init__(self, host, desc, parent, specificID)
		self.domain = domain
		self.port = port
	def Run(self, parentXML):
		"""
		Run the test
		
		Add a "test" node to parentXML with the attributes:
			id: the test ID
			status: -1 => error, 0 => warning, 1: success
			msg: an additional string that will be displayed on the UI
			
		Return the newly created node
		"""
		child = NetworkTest.Run(self, parentXML)
		try:
			try:
				svr = smtplib.SMTP(host=self.host, port = self.port)
			except Exception, details:
				raise RuntimeError, "Unable to connect (%s)" % str(details)
			try:
				code, msg = svr.helo(self.domain)
			except Exception, details:
				raise RuntimeError, "Unable to send a HELO command (%s)" % str(details)
			child.setProp("status", "1")
			child.setProp("msg", "> HELO %s\n< %03d %s" % (self.domain, code, msg))
		except Exception, details:
			child.setProp("status", "-1")
			child.setProp("msg", str(details))
		return child

class LSTest(NetworkTest):
	"""
	This test try a "HELO" on an Load Sharing server
	"""
	DEFAULT_PORT = 789
	"""The default port for LS services"""
	TIMEOUT = 5
	"""The default timeout in seconds"""
	def __init__(self, host, desc, parent, port = DEFAULT_PORT, specificID = ''):
		"""
		Constructor
		Params:
			host: the LS server address
			desc: the test description
			parent: the "parent" test
			port: the LS server port
			specificID: the string to add to the class name in order to get a unique ID
		"""
		NetworkTest.__init__(self, host, desc, parent, specificID)
		self.port = port
	def Run(self, parentXML):
		"""
		Run the test
		
		Add a "test" node to parentXML with the attributes:
			id: the test ID
			status: -1 => error, 0 => warning, 1: success
			msg: an additional string that will be displayed on the UI
			
		Return the newly created node
		"""
		child = NetworkTest.Run(self, parentXML)
		try:
			try:
				svr = telnetlib.Telnet(host=self.host, port = self.port)
			except Exception, details:
				raise RuntimeError, "Unable to connect (%s)" % str(details)
			try:
				svr.write("get version\r\n\r\n")
				reply = svr.read_until("\n", LSTest.TIMEOUT)[:-1]
				svr.close()
			except Exception, details:
				raise RuntimeError, "Unable to send a VERSION command (%s)" % str(details)
			child.setProp("status", "1")
			child.setProp("msg", "> get version\n< %s" % (reply))
		except Exception, details:
			child.setProp("status", "-1")
			child.setProp("msg", str(details))
		return child

class FtpTest(NetworkTest):
	class Params:
		def __init__(self, xmlNode, file):
			"""
			Constructor
			
			params:
				xmlNode: the xml element where all the settings are
				file: the configuration file name
			"""
			self.enabled = False
			self.server = None
			self.port = None
			self.dir = None
			self.user = None
			self.passwd = None
			self.proxy_server = None
			self.proxy_port = None
			self.proxy_user = None
			self.proxy_passwd = None
			self.file = file
			try:
				self.enabled = (int(xmlNode.xpathEval("Attr[@name='Enabled']/@value")[0].content) != 0)
			except:
				raise RuntimeError, _TR_("T_AVSPAM_NOENABLE", "AV/SPAM Update: Invalid or missing enable flag")
			if not self.enabled:
				return
			try:
				self.server = xmlNode.xpathEval("Attr[@name='Server']/@value")[0].content.strip()
				if len(self.server) == 0:
					raise RuntimeError, _TR_("T_AVSPAM_NOSERVER", "AV/SPAM Update: Invalid or missing update server")
			except:
				raise RuntimeError, _TR_("T_AVSPAM_NOSERVER", "AV/SPAM Update: Invalid or missing update server")
			try:
				self.port = int(xmlNode.xpathEval("Attr[@name='Port']/@value")[0].content)
				if self.port <= 0:
					raise RuntimeError, _TR_("T_AVSPAM_INVPORT", "AV/SPAM Update: Invalid port number")
			except:
				self.port = 21
			try:
				self.dir = xmlNode.xpathEval("Attr[@name='SourcePath']/@value")[0].content.strip()
				if len(self.dir) == 0:
					raise RuntimeError, _TR_("T_AVSPAM_NODIR", "AV/SPAM Update: Invalid or missing server directory")
			except:
				raise RuntimeError, _TR_("T_AVSPAM_NODIR", "AV/SPAM Update: Invalid or missing server directory")
			try:
				self.user = xmlNode.xpathEval("Attr[@name='Username']/@value")[0].content.strip()
				if len(self.user) == 0:
					self.user = "anonymous"
			except:
				self.user = "anonymous"
			try:
				self.passwd = xmlNode.xpathEval("Attr[@name='Password']/@value")[0].content.strip()
				if len(self.passwd) == 0:
					self.passwd = "anonymous"
			except:
				self.passwd = "anonymous"
			# proxy
			try:
				self.proxy_server = xmlNode.xpathEval("Attr[@name='ProxyServer']/@value")[0].content.strip()
				if len(self.proxy_server) == 0:
					self.proxy_server = None
			except:
				pass
			if self.proxy_server is None:
				return
			try:
				self.proxy_port = int(xmlNode.xpathEval("Attr[@name='ProxyPort']/@value")[0].content)
				if self.proxy_port <= 0:
					raise RuntimeError, _TR_("T_AVSPAM_INVPROXYPORT", "AV/SPAM Update: Invalid proxy port number")
			except:
				self.proxy_port = 21
			try:
				self.proxy_user = xmlNode.xpathEval("Attr[@name='ProxyUsername']/@value")[0].content.strip()
				if len(self.proxy_user) == 0:
					self.proxy_user = None
			except:
				pass
			try:
				self.proxy_passwd = xmlNode.xpathEval("Attr[@name='ProxyPassword']/@value")[0].content.strip()
				if len(self.proxy_passwd) == 0:
					self.proxy_passwd = None
			except:
				pass
	"""
	This test try to retrieve a file on a FTP server
	"""
	def Run(self, parentXML):
		"""
		Run the test
		
		Add a "test" node to parentXML with the attributes:
			id: the test ID
			status: -1 => error, 0 => warning, 1: success
			msg: an additional string that will be displayed on the UI
			
		Return the newly created node
		"""
		child = NetworkTest.Run(self, parentXML)
		if ((self.host.proxy_server is None) or \
			((self.host.proxy_server != None) and (self.host.proxy_user is None)  and (self.host.proxy_port == self.host.port))):
			# "Normal" ftp session
			def FTPCallback(bytes):
				pass
			# params
			if self.host.proxy_server is None:
				f_host = self.host.server
				f_user = self.host.user
				proxy = "no proxy"
			else:
				f_host = self.host.proxy_server
				f_user = self.host.user + "@" + self.host.server
				proxy = "proxy: "+self.host.proxy_server
			# connect
			passive = "passive mode"
			try:
				try:
					ftp = ftplib.FTP(host=f_host, user=f_user, passwd=self.host.passwd)
				except Exception, details:
					raise RuntimeError, "Unable to connect (%s)" % str(details)
				try:
					ftp.cwd(self.host.dir)
				except Exception, details:
					raise RuntimeError, "Unable to go to the directory '%s' (%s)" % (self.host.dir, str(details))
				try:
					ftp.set_pasv(True)
					ftp.retrbinary('RETR %s' % self.host.file, FTPCallback)
				except:
					try:
						passive = "active mode"
						ftp.set_pasv(False)
						ftp.retrbinary('RETR %s' % self.host.file, FTPCallback)
					except Exception, details:
						raise RuntimeError, "Unable to get the file '%s' (%s)" % (self.host.file, str(details))
			except Exception, details:
				child.setProp("status", "-1")
				child.setProp("msg", str(details))
				return child
			child.setProp("status", "1")
			child.setProp("msg", "Got the file 'ftp://%s:XXX@%s/%s/%s' (%s, %s)" % 
				(self.host.user, self.host.server, self.host.dir, self.host.file, passive, proxy))
		else:
			# proxy with a username and password
			os.environ["ftp_proxy"] = "%s:%d" % (self.host.proxy_server, self.host.proxy_port)
			cmd = "/usr/bin/wget --timeout=30 -O - "
			if (self.host.proxy_user != None):
				cmd += "--proxy-user=%s " % self.host.proxy_user
			if (self.host.proxy_passwd != None):
				cmd += "--proxy-passwd=%s " % self.host.proxy_passwd
			cmd += "ftp://%s:%s@%s%s" % (self.host.user, self.host.passwd, self.host.server, os.path.join('/', os.path.join(self.host.dir, self.host.file)))
			Trace("Launching '%s'" % cmd) 
			status, output = NewGetStatusOutput(cmd)
			Trace("wget done")
			if status == 0:
				child.setProp("status", "1")
				child.setProp("msg", "Got the file 'ftp://%s:XXX@%s/%s/%s'" % 
					(self.host.user, self.host.server, self.host.dir, self.host.file))
			else:
				child.setProp("status", "-1")
				child.setProp("msg", "Unable to get the file 'ftp://%s:XXX@%s/%s/%s'\n%s" % 
					(self.host.user, self.host.server, self.host.dir, self.host.file, output))
			os.environ["ftp_proxy"] = ""
		return child
			
class EpoInstallTest(CTest):
	"""
	Test the installation of the EPO stuff
	"""
	def __init__(self, parent, specificID = ''):
		"""
		Constructor
		Params:
			parent: the "parent" test
			specificID: the string to add to the class name in order to get a unique ID
		"""
		CTest.__init__(self, parent, specificID)
	def getDesc(self):
		"""Returns the test description"""
		return _TR_("T_EPO_CLIENT_INSTALL_TEST", "Check EPO agent installation")
	def Run(self, parentXML):
		"""
		Run the test
		
		Add a "test" node to parentXML with the attributes:
			id: the test ID
			status: -1 => error, 0 => warning, 1: success
			msg: an additional string that will be displayed on the UI
			
		Return the newly created node
		"""
		child = CTest.Run(self, parentXML)
		msg = ''
		status = "1"
		for f in [ '/opt/NETAepoagt/bin/naimas32', '/opt/NETAepoagt/lib/libnanettcp.so' ]:
			if len(msg) != 0:
				msg += '\n'
			msg += "Checking the existence of '%s': " % f
			if os.path.isfile(f):
				msg += "OK"
			else:
				msg += "not found"
				status = "-1"
		child.setProp("status", status)
		child.setProp("msg", msg)
		return child

class EpoRunningTest(CTest):
	"""
	Test if the EPO processes are running
	"""
	def __init__(self, parent, specificID = ''):
		"""
		Constructor
		Params:
			parent: the "parent" test
			specificID: the string to add to the class name in order to get a unique ID
		"""
		CTest.__init__(self, parent, specificID)
	def getDesc(self):
		"""Returns the test description"""
		return _TR_("T_EPO_CLIENT_PROCESS_TEST", "Check EPO agent processes")
	def Run(self, parentXML):
		"""
		Run the test
		
		Add a "test" node to parentXML with the attributes:
			id: the test ID
			status: -1 => error, 0 => warning, 1: success
			msg: an additional string that will be displayed on the UI
			
		Return the newly created node
		"""
		child = CTest.Run(self, parentXML)
		msg = ''
		status = "1"
		for f in [ '/var/lock/subsys/ws_nwa' ]:
			if len(msg) != 0:
				msg += '\n'
			msg += "Checking the existence of '%s': " % f
			if os.path.isfile(f):
				msg += "OK"
			else:
				msg += "not found"
				status = "-1"
		for p in [ 'naimas32', 'epo_pp' ]:
			if len(msg) != 0:
				msg += '\n'
			msg += "Checking the existence of '%s' process: " % p
			cmd = "killall -q -0 " + p
			Trace("Running killall test '%s'" % cmd)
			cstatus, res = NewGetStatusOutput(cmd)
			Trace("Test done: (%d, %s)" % (cstatus, res))
			if cstatus == 0:
				msg += "OK"
			else:
				msg += "not found"
				status = "-1"
		child.setProp("status", status)
		child.setProp("msg", msg)
		return child

class EpoListeningTest(CTest):
	"""
	Test if the EPO agent is listening
	"""
	def __init__(self, parent, specificID = ''):
		"""
		Constructor
		Params:
			parent: the "parent" test
			specificID: the string to add to the class name in order to get a unique ID
		"""
		CTest.__init__(self, parent, specificID)
	def getDesc(self):
		"""Returns the test description"""
		return _TR_("T_EPO_CLIENT_LISTEN_TEST", "Check EPO agent port")
	def Run(self, parentXML):
		"""
		Run the test
		
		Add a "test" node to parentXML with the attributes:
			id: the test ID
			status: -1 => error, 0 => warning, 1: success
			msg: an additional string that will be displayed on the UI
			
		Return the newly created node
		"""
		child = CTest.Run(self, parentXML)
		cmd = "netstat -tlnp  | grep '\<naimas32\>'"
		Trace("Running listen epo test '%s'" % cmd)
		cstatus, res = NewGetStatusOutput(cmd)
		Trace("Test done: (%d, %s)" % (cstatus, res))
		if cstatus == 0:
			child.setProp("status", "1")
			child.setProp("msg", res)
		else:
			child.setProp("status", "-1")
			child.setProp("msg", "no 'maimas32' process is listening on a TCP port")
		return child

class GatewayTestGroup(TestGroup):
	"""
	This groups all the tests related to the gateway
	"""
	def __init__(self, parent, specificID = ''):
		"""
		Constructor
		Params:
			parent: the "parent" test
			specificID: the string to add to the class name in order to get a unique ID
		"""
		TestGroup.__init__(self, parent, specificID)
		# get the settings
		machine = self.getDoc(CTest.MACHINE)
		ctxt = machine.xpathNewContext()
		gateway = ctxt.xpathEval("//Settings[@name='network-settings']/Attr[@name='IPDefaultRouter']/@value")
		if (len(gateway) != 1) :
			raise _TR_("T_GETAWAY_IADD", "Unable to get the gateway address")
		self.gateway = gateway[0].content
		# set up the tests
		self.tests.append(PingTest(self.gateway, _TR_("T_GETAWAY_PING", "Ping the gateway"), self))
		self.tests.append(ArpTest(self.gateway, _TR_("T_GETAWAY_ARP", "Look for the gateway in the ARP table"), self))
	def getDesc(self):
		return _TR_("T_GETAWAY_DESC", "Gateway (%s)") % self.gateway

class DNSTestGroup(TestGroup):
	"""
	This groups all the tests related to DNS servers
	"""
	def __init__(self, parent, specificID = ''):
		"""
		Constructor
		Params:
			parent: the "parent" test
			specificID: the string to add to the class name in order to get a unique ID
		"""
		TestGroup.__init__(self, parent, specificID)
		# get the settings
		network = self.getDoc(CTest.NETWORK)
		ctxt = network.xpathNewContext()
		dns = ctxt.xpathEval("//Settings[@name='network-settings']/NetObjectList[@name='NameServers']/NetObject/Address/@value")
		# set up the tests
		i = 0
		for svr in dns:
			asvr = svr.content
			self.tests.append(PingTest(asvr, _TR_("T_DNS_PING", "Ping the DNS server '%s'") % asvr, self, specificID = "_"+str(i)))
			self.tests.append(DigTest(asvr, _TR_("T_DNS_DIG", "Query the DNS server '%s'") % asvr, self, specificID = "_"+str(i)))
			i += 1
	def getDesc(self):
		return _TR_("T_DNS_DESC", "DNS servers")

class ApplianceNameAndDomainTestGroup(TestGroup):
	"""
	This groups all the tests related to DNS servers
	"""
	def __init__(self, parent, specificID = ''):
		"""
		Constructor
		Params:
			parent: the "parent" test
			specificID: the string to add to the class name in order to get a unique ID
		"""
		TestGroup.__init__(self, parent, specificID)
		# get the settings
		machine = self.getDoc(CTest.MACHINE)
		ctxt = machine.xpathNewContext()
		appName = ctxt.xpathEval("//Settings[@name='network-settings']/Attr[@name='IPHostname']/@value")
		if (len(appName) != 1) :
			raise _TR_("T_ANDN_NON", "Unable to get the appliance's name")
		appName = appName[0].content
		appDomain = ctxt.xpathEval("//Settings[@name='network-settings']/Attr[@name='IPDomainName']/@value")
		# the domain can be null...
		if (len(appDomain) == 1) :
			appDomain = appDomain[0].content
			if len(appDomain) == 0:
				appDomain = None
		else:
			appDomain = None
		# no domain name => no DNS checks
		if appDomain != None:
			appName +=  "." + appDomain
			# set up the tests
			self.tests.append(DigTest(None, _TR_("T_APPL_DN_DIG", "Query the appliance domain name '%s'") % appDomain, self, "_DN",
				query_type = DigTest.T_NS, test_address = appDomain))
			self.tests.append(DigTest(None, _TR_("T_APPL_NM_DIG", "Query the appliance address '%s'") % appName, self, "_NM",
				query_type = DigTest.T_A, test_address = appName))
	def getDesc(self):
		return _TR_("T_ANDN_DESC", "Appliance name and domain")

class StaticRoutesTestGroup(TestGroup):
	"""
	This groups all the tests related to static routes
	"""
	def __init__(self, parent, specificID = ''):
		"""
		Constructor
		Params:
			parent: the "parent" test
			specificID: the string to add to the class name in order to get a unique ID
		"""
		TestGroup.__init__(self, parent, specificID)
		# get the settings
		machine = self.getDoc(CTest.MACHINE)
		ctxt = machine.xpathNewContext()
		dns = ctxt.xpathEval("//Settings[@name='network-settings']/NetObjectList[@name='StaticRoutes']/NetObject/Gateway/@value")
		# set up the tests
		i = 0
		for svr in dns:
			asvr = svr.content
			self.tests.append(PingTest(asvr, _TR_("T_STROUTES_PING", "Ping the gateway %s") % asvr, self, specificID = "_"+str(i)))
			self.tests.append(ArpTest(asvr, _TR_("T_STROUTES_ARP", "Look for the gateway %s in the ARP table") % asvr, self, specificID = "_"+str(i)))
			i += 1
	def getDesc(self):
		return _TR_("T_STROUTES_DESC", "Static Routes")

class UpdateTestGroup(TestGroup):
	"""
	This groups all the tests related to updates
	"""
	def __init__(self, parent, specificID = ''):
		"""
		Constructor
		Params:
			parent: the "parent" test
			specificID: the string to add to the class name in order to get a unique ID
		"""
		TestGroup.__init__(self, parent, specificID)
		# get the settings and set up the tests
		system = self.getDoc(CTest.SYSTEM)
		ctxt = system.xpathNewContext()
		# AV
		av_node = ctxt.xpathEval("//Settings[@name='anti-virus-updates']")
		if (len(av_node) != 1) :
			raise _TR_("T_AV_UPD_XML", "Unable to get the anti-virus update settings")
		av_settings = FtpTest.Params(av_node[0], "update.ini")
		self.tests.append(FtpTest(av_settings, _TR_("T_UPDATE_AV", "Talk to the AV update server"), self, specificID='_AV'))
		# SPAM
		spam_node = ctxt.xpathEval("//Settings[@name='anti-spam-updates']")
		if (len(spam_node) != 1) :
			raise _TR_("T_SPAM_UPD_XML", "Unable to get the anti-spam update settings")
		spam_settings = FtpTest.Params(spam_node[0], "spamupd30.ini")
		self.tests.append(FtpTest(spam_settings, _TR_("T_UPDATE_SPAM", "Talk to the SPAM update server"), self, specificID='_SPAM'))
	def getDesc(self):
		return _TR_("T_UPDATES_DESC", "Updates")
		
class SMTPRelaysTestGroup(TestGroup):
	"""
	This groups all the tests related to updates
	"""
	def __init__(self, parent, type, specificID = ''):
		"""
		Constructor
		Params:
			parent: the "parent" test
			type: the name to look for in smtp.xml
			specificID: the string to add to the class name in order to get a unique ID
		"""
		TestGroup.__init__(self, parent, specificID)
		# get the settings
		smtp = self.getDoc(CTest.SMTP)
		ctxt = smtp.xpathNewContext()
		relay_nodes = ctxt.xpathEval("//Settings[@name='delivery']/List[@name='%s']/Attr/@value" % type)
		if (len(relay_nodes) != 0) :
			# build a list of servers
			svrs = {}
			for svr_list in relay_nodes:
				# NB: the 1st element in the list is RELAYLIST and the second the domain
				for svr in svr_list.content.split()[2:]:
					svrs[svr] = True
			# set up the tests
			i = 0
			for svr in svrs.iterkeys():
				self.tests.append(PingTest(svr, _TR_("T_SMTP_PING", "Ping the SMTP server: %s") % svr, self, specificID = "_"+str(i)))
				self.tests.append(SmtpTest(svr, _TR_("T_SMTP_TEST", "Talk to the SMTP server: %s") % svr, self, specificID = "_"+str(i)))
				i += 1
		
class DomainRelaysTestGroup(SMTPRelaysTestGroup):
	def __init__(self, parent, specificID = ''):
		"""
		Constructor
		Params:
			parent: the "parent" test
			specificID: the string to add to the class name in order to get a unique ID
		"""
		SMTPRelaysTestGroup.__init__(self, parent, "DomainRelay", specificID)
	def getDesc(self):
		return _TR_("T_DOMAINRELAYS_DESC", "Domain Relays")

class FallbackRelaysTestGroup(SMTPRelaysTestGroup):
	def __init__(self, parent, specificID = ''):
		"""
		Constructor
		Params:
			parent: the "parent" test
			specificID: the string to add to the class name in order to get a unique ID
		"""
		SMTPRelaysTestGroup.__init__(self, parent, "FallbackRelay", specificID)
	def getDesc(self):
		return _TR_("T_FALLBACKRELAYS_DESC", "Fallback Relays")

class LSTestGroup(TestGroup):
	"""
	This groups all the tests related to load-sharing servers
	"""
	def __init__(self, parent, specificID = ''):
		"""
		Constructor
		Params:
			parent: the "parent" test
			specificID: the string to add to the class name in order to get a unique ID
		"""
		TestGroup.__init__(self, parent, specificID)
		# get the settings
		machine = self.getDoc(CTest.MACHINE)
		ctxt = machine.xpathNewContext()
		ls_enable = ctxt.xpathEval("//Settings[@name='ls-settings']/Attr[@name='Enabled']/@value")
		if (len(ls_enable) != 1):
			raise _TR_("T_LS_XML", "Unable to get the load sharing settings")
		ls_enable = (int(ls_enable[0].content) != 0)
		if ls_enable:
			ls_nodes = ctxt.xpathEval("//Settings[@name='ls-settings']/NetObjectList[@name='Servers']/NetObject[@type='LOADSERVER']/Address/@value")
			if (len(ls_nodes) != 0):
				# set up the tests
				i = 0
				for asvr in ls_nodes:
					svr = asvr.content
					self.tests.append(PingTest(svr, _TR_("T_LS_PING", "Ping the Load Sharing server: %s") % svr, self, specificID = "_"+str(i)))
					self.tests.append(LSTest(svr, _TR_("T_LS_TEST", "Talk to the Load Sharing server: %s") % svr, self, specificID = "_"+str(i)))
					i += 1
	def getDesc(self):
		return _TR_("T_LS_DESC", "Load sharing")

class EPOTestGroup(TestGroup):
	def __init__(self, parent, specificID = ''):
		"""
		Constructor
		Params:
			parent: the "parent" test
			specificID: the string to add to the class name in order to get a unique ID
		"""
		TestGroup.__init__(self, parent, specificID)
		# get the settings
		machine = self.getDoc(CTest.MACHINE)
		ctxt = machine.xpathNewContext()
		epo_active = ctxt.xpathEval("//Settings[@name='epoagent']/Attr[@name='Enabled']/@value")
		if len(epo_active) == 1 and epo_active[0].content == "1":
			self.tests.append(EpoInstallTest(self))
			self.tests.append(EpoRunningTest(self))
			self.tests.append(EpoListeningTest(self))
	def getDesc(self):
		return _TR_("T_EPO_CLIENT_DESC", "EPO agent")
		
def Run():
	global formatXML

	try:
		# default timeout: 60s
		timeoutsocket.setDefaultSocketTimeout(60)

		args, params = libwsa.Init()
	
		####################################
		# parse the arguments
		####################################
		ctxt = args.xpathNewContext()

		testID = ctxt.xpathEval("/command/arg[@name='test']/@value")
		if len(testID) == 1:
			testID = testID[0].content
		else:
			testID = None
		####################################
		# build the list of tests
		####################################
		tests = [
			GatewayTestGroup(None),
			DNSTestGroup(None),
			ApplianceNameAndDomainTestGroup(None),
			StaticRoutesTestGroup(None),
			DomainRelaysTestGroup(None),
			FallbackRelaysTestGroup(None),
			LSTestGroup(None),
			UpdateTestGroup(None),
			EPOTestGroup(None)
			]
	
		####################################
		# do the test and build the reply
		####################################
		reply = libxml2.newDoc("1.0")
		root = reply.newChild(None, "res", None)
		root.setProp("status", "0")
		#
		if testID is None:
			#------------------------
			# build a test list
			#------------------------
			output = root.newChild(None, "output", None)
			output.setProp("name", "test-list")
			for test in tests:
				test.AddDescTo(output)
		else:
			#------------------------
			# run the test
			#------------------------
			output = root.newChild(None, "output", None)
			output.setProp("name", "results")
			found = False
			for test in tests:
				if test.RunIf(testID, output):
					found = True
					break
			if not found:
				raise _TR_("T_BAD_TEST_ID", "Unable to found a test with the ID: '%s'") % testID
		####################################
		# we're done!
		####################################
		print root.serialize(format=formatXML)
		sys.exit(0)
	except SystemExit:
		pass
	except:
		libwsa.PrintError(sys.exc_info()[1])
		sys.exit(-1)

def Test():
	""" Test function"""
	cmd1 = \
	"""
<command name="WSConfigTest" id="1">
</command>
	"""
	cmd2 = \
	"""
<command name="WSConfigTest" id="1">
	<arg name="test" value="GatewayTestGroup:PingTest"/>
</command>
	"""
	cmd3 = \
	"""
<command name="WSConfigTest" id="1">
	<arg name="test" value="GatewayTestGroup:ArpTest"/>
</command>
	"""
	cmd4 = \
	"""
<command name="WSConfigTest" id="1">
	<arg name="test" value="EPOTestGroup:EpoListeningTest"/>
</command>
	"""
	cmd = cmd4
	params = \
	"""
<params>
</params>
	"""
	sys.argv = [ sys.argv[0], cmd, params ]
	Run()


if __name__ == '__main__':
	Run()
