#!/usr/bin/python
# Copyright (C) 2003 Networks Associates Technology Inc.
# All rights reserved.

"""
Utility routines for the python scripts calls by the wsadmin Apache module.
"""

import os, sys, libxml2, popen2, types, gettext, signal

__all__ = ["do_trace", "tmp_dir", "Init", "RunCmd"]

# flag for Trace
do_trace = False
# temporary directory
tmp_dir = os.getenv('WSADMIN_TMP_DIR', '/tmp/wsa')
# size of buffers
BUFSIZE= 1024
# set to true if we are signaling the whole process group
SIGNALING = False

def xmlErrHandler(ctx, str):
	"""error handler for libxml2"""
	pass

def _TR_(ID, msg):
	"""Translate a message"""
	return _(msg)

def InitTransl():
	"""Initialize gettext"""
	lang = gettext.translation(
		domain=os.getenv("WS_TEXTDOMAIN", "wsadmin"),
		localedir=os.getenv("WS_TEXTDOMAINDIR", "/opt/NETAwss/ui/wsadmin/locale"), 
		languages=[os.getenv("WS_LANGUAGE", "en_US.UTF-8")],
		fallback = True
		)
	lang.install()

def SignalHandler():
	# if the effective user id is root, we need to "relay" the signal from Apache 
	# to the other processes in the group as they have been launched with the RUID 
	# set to root in order to allow bash scripts to work
	if (os.geteuid() == 0) and (not SIGNALING):
		SIGNALING = True
		os.kill(-os.getgid(), signal.SIGTERM)
		SIGNALING = False

def Init():
	"""
	Common initialization routine for all the scripts

	Returns (args, params) where:
		args is the xml documents contening the contents of the command
		params is the xml documents contening the contents of the parameters
	If an error occurs, an exception is thrown with the appropriate error message
	"""
	####################################
	# initialize gettext
	####################################
	InitTransl()
	####################################
	# register the error handler
	####################################
	libxml2.registerErrorHandler(xmlErrHandler, "")
	####################################
	# register the signal handler
	####################################
	signal.signal(signal.SIGTERM, SignalHandler)
	####################################
	# check and parse the arguments
	####################################
	if len(sys.argv) != 3:
		raise RuntimeError, _TR_("WSS_ENA", 'Invalid number of arguments')
	try:
		args_doc = libxml2.parseDoc(sys.argv[1])
	except:
		raise RuntimeError, _TR_("WSS_IVS", 'Invalid "vars" script argument')
	try:
		params_doc = libxml2.parseDoc(sys.argv[2])
	except:
		raise RuntimeError, _TR_("WSS_IPS", 'Invalid "params" script argument')
	return (args_doc, params_doc)

def PrintError(msg):
	"""
	Print an error message in the XML format, adding any relevant information from the last exception thrown

	Arguments:
		msg: the main message to display
	"""
	from xml.sax.saxutils import quoteattr
	from traceback import print_exc
	import StringIO

	if msg is None:
		try:
			# try to deduce it from the exception arguments
			msg = sys.exc_info()[0]
		except:
			pass

	out = StringIO.StringIO()
	print_exc(None, out)
	print "<res status='-1'><error message=%s><![CDATA[\n%s\n]]></error></res>" % (quoteattr(str(msg)), str(out.getvalue()))

def PrintErrorBase(msg, ctx):
	"""
	Print an error message in the XML format, adding any relevant information from the last exception thrown

	Arguments:
		msg: the main message to display
	"""
	from xml.sax.saxutils import quoteattr
	
	print "<res status='-1'><error message=%s><![CDATA[\n%s\n]]></error></res>" % (quoteattr(str(msg)), ctx)

def Trace(msg):
	"""
	Print a message into the messages log if libwsa.do_trace is set to true

	Arguments:
		msg: the message to put in the logs
	"""
	global do_trace
	if do_trace:
		import syslog
		syslog.syslog(syslog.LOG_ERR, msg)

def run_child(self, cmd):
	"""This function is an exact copy of popen2.Popen3._run_child
	at the only exception that the file descriptor 3 is not closed"""
	if isinstance(cmd, types.StringTypes):
		cmd = ['/bin/sh', '-c', cmd]
	for i in range(4, popen2.MAXFD):
		try:
			os.close(i)
		except:
			pass
	# if the effective user id os root, we need to set also the real user ID 
	# as Bash will reset the EUID to the RUID
	# ===> this means that only this script will receive TERM signals from Apache 
	#           and that we have to relay them to all the processes in the group
	if os.geteuid() == 0:
		try:
			os.setreuid(0, 0)
			os.setregid(0, 0)
		except:
			# just in case...
			pass
	try:
		os.execvp(cmd[0], cmd)
	finally:
		os._exit(1)

# override popen2.Popen3._run_child with run_child
popen2.Popen3._run_child = run_child

def RunCmd(cmd, callback, timeout = 1, initCallback = None):
	"""
	Launch a command and call a callback at regular interval time

	Arguments:
		cmd: the command to launch:
			it can be a list of the form [cmd, arg1 , arg2, .... ]. ex: ["/usr/bin/tar", "tvf", "test.tar"]
			it can bin a string, in this case a shell will be launched for evaluating the string. ex: "/bin/ls *.cpp"
		callback: the callback function. The callback must have the following signature:
			bool f(read) where read is a list of 2 strings:
				read[0] is what has been read on the standard input or None if nothing has been read
				read[1] is what has been read on the standard error or None if nothing has been read
			the callback function returns true if it want to continue false otherwise
		timeout: timeout is the interval in seconds at which to call the callback
		initCallback: if not NULL, this callback is called once the process has been launched with the PID of the child as argument
	"""
	import select, signal, errno

	setattr(popen2.Popen3, "_run_child", run_child)
	timeout *= 1000
	child = popen2.Popen3(cmd, True)
	if (child.poll() > 0):
		raise RuntimeError, _TR_("WSS_ERL", "Unable to launch '%s'") % (cmd)
		
	if initCallback != None:
		initCallback(child.pid)
		
	fl = { \
		child.fromchild.fileno() : 0, \
		child.childerr.fileno() : 1 \
		}
	p = select.poll()
	for f in fl.keys():
		p.register(f, select.POLLIN)

	poll_loop = True
	while (poll_loop):
		try:
			ready = p.poll(timeout)
		except select.error, v:
			if v[0] != errno.EINTR:
				raise
			continue
		res = [ None, None ]
		got_something = False
		for fd, mode in ready:
			if mode & select.POLLIN:
				res[ fl[fd] ] = os.read(fd, BUFSIZE)
				got_something = True
		if (not callback(res)):
			# the callback told us to abort
			os.kill(child.pid, signal.SIGTERM)
			return -1
		# we received nothing, is the command finished ?
		if not got_something:
		   poll_loop = (child.poll() < 0)
	return child.poll()

def NewGetStatusOutput(cmd):
	class NGSOCallback:
		def __init__(self):
			self.output = ""
		def Callback(self, res):
			if res[0] != None:
				self.output += res[0]
			# stderr should be empty, but just in case
			if res[1] != None:
				self.output += res[1]
			return True
	output = NGSOCallback()
	status = RunCmd('{ ' + cmd + '; } 2>&1', output.Callback)
	return (status, output.output)

def CheckPassword(user, password, user_file):
	"""
	Check that a user has given the correct password.
	
	Arguments:
		user: the user name
		password: the password
		user_file: the path to the XML user file
	"""
	import md5
	
	# load the users file
	try:
		users_doc = libxml2.parseFile(user_file)
	except:
		raise RuntimeError, _TR_("WSS_IUF", 'Unable to read or parse the users file')
	# get the hex digest of the password
	ctxt = users_doc.xpathNewContext()
	truePassword = ctxt.xpathEval("/users/user[@name='%s']/@password" % user)
	if len(truePassword) != 1:
		raise RuntimeError, _TR_("WSS_IUP", 'Invalid or missing password for the user %s') % user
	truePassword = truePassword[0].content
	m = md5.new(password)
	return m.hexdigest() == truePassword;


def Log(event, reason = None, info = []):
	"""
	Log an event
	
	Arguments:
		event: the event name
		reason: the reason name
		info: a list of (Propert_Name, Property_Value)
	"""
	try:
		WSBIN = os.environ["WSBIN"]
	except:
		WSBIN = "/opt/NETAwss/bin"
	
	try:
		WSLIBDIR = os.environ["WSLIBDIR"]
	except:
		WSLIBDIR = "/opt/NETAwss/lib"
	
	evrep = "LD_LIBRARY_PATH="+WSLIBDIR+" "+os.path.join(WSBIN, "evrep")
	evrep += " -n UI -e " + event
	if (reason != None):
		evrep += "::" + reason
	evrep += " "
	try:
		info.append( ("UI_SESSION_ID", os.environ["WS_SESSION"]) )
		info.append( ("USER_NAME", os.environ["WS_USER_ID"]) )
		info.append( ("SOURCE_IP", os.environ["WS_SOURCE_IP"]) )
	except:
		pass
	for name, value in info:
		evrep += "-i %s -v '%s' " % (name, value)
	NewGetStatusOutput(evrep)
